/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */
/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 * Portions Copyright 2007-2011 Apple Inc.
 */
typedef uint64_t resolver_result_t;


enum path_operation    {
    OP_LOOKUP,
    OP_MOUNT,
    OP_UNMOUNT,
    OP_STATFS,
    OP_OPEN,
    OP_LINK,
    OP_UNLINK,
    OP_RENAME,
    OP_CHDIR,
    OP_CHROOT,
    OP_MKNOD,
    OP_MKFIFO,
    OP_SYMLINK,
    OP_ACCESS,
    OP_PATHCONF,
    OP_READLINK,
    OP_GETATTR,
    OP_SETATTR,
    OP_TRUNCATE,
    OP_COPYFILE,
    OP_MKDIR,
    OP_RMDIR,
    OP_REVOKE,
    OP_EXCHANGEDATA,
    OP_SEARCHFS,
    OP_FSCTL,
    OP_GETXATTR,
    OP_SETXATTR,
    OP_REMOVEXATTR,
    OP_LISTXATTR,
    OP_MAXOP    /* anything beyond previous entry is invalid */
};

/*!
 @typedef trigger_vnode_resolve_callback_t
 @abstract function prototype for a trigger vnode resolve callback
 @discussion This function is associated with a trigger vnode during a vnode create.  It is
 typically called when a lookup operation occurs for a trigger vnode
 @param vp The trigger vnode which needs resolving
 @param cnp Various data about lookup, e.g. filename and state flags
 @param pop The pathname operation that initiated the lookup (see enum path_operation).
 @param flags
 @param data Arbitrary data supplied by vnode trigger creator
 @param ctx Context for authentication.
 @return RESOLVER_RESOLVED, RESOLVER_NOCHANGE, RESOLVER_UNRESOLVED or RESOLVER_ERROR
 */
typedef resolver_result_t (* trigger_vnode_resolve_callback_t)(
                                                               vnode_t                vp,
                                                               const struct componentname *    cnp,
                                                               enum path_operation        pop,
                                                               int                flags,
                                                               void *                data,
                                                               vfs_context_t            ctx);

/*!
 @typedef trigger_vnode_unresolve_callback_t
 @abstract function prototype for a trigger vnode unresolve callback
 @discussion This function is associated with a trigger vnode during a vnode create.  It is
 called to unresolve a trigger vnode (typically this means unmount).
 @param vp The trigger vnode which needs unresolving
 @param flags Unmount flags
 @param data Arbitrary data supplied by vnode trigger creator
 @param ctx Context for authentication.
 @return RESOLVER_NOCHANGE, RESOLVER_UNRESOLVED or RESOLVER_ERROR
 */
typedef resolver_result_t (* trigger_vnode_unresolve_callback_t)(
                                                                 vnode_t        vp,
                                                                 int        flags,
                                                                 void *        data,
                                                                 vfs_context_t    ctx);

/*!
 @typedef trigger_vnode_rearm_callback_t
 @abstract function prototype for a trigger vnode rearm callback
 @discussion This function is associated with a trigger vnode during a vnode create.  It is
 called to verify a rearm from VFS (i.e. should VFS rearm the trigger?).
 @param vp The trigger vnode which needs rearming
 @param flags
 @param data Arbitrary data supplied by vnode trigger creator
 @param ctx Context for authentication.
 @return RESOLVER_NOCHANGE or RESOLVER_ERROR
 */
typedef resolver_result_t (* trigger_vnode_rearm_callback_t)(
                                                             vnode_t        vp,
                                                             int        flags,
                                                             void *        data,
                                                             vfs_context_t    ctx);

/*!
 @typedef trigger_vnode_reclaim_callback_t
 @abstract function prototype for a trigger vnode reclaim callback
 @discussion This function is associated with a trigger vnode during a vnode create.  It is
 called to deallocate private callback argument data
 @param vp The trigger vnode associated with the data
 @param data The arbitrary data supplied by vnode trigger creator
 */
typedef void (* trigger_vnode_reclaim_callback_t)(
                                                  vnode_t        vp,
                                                  void *        data);

/*!
 @function vnode_trigger_update
 @abstract Update a trigger vnode's state.
 @discussion This allows a resolver to notify VFS of a state change in a trigger vnode.
 @param vp The trigger vnode whose information to update.
 @param result A compound resolver result value
 @return EINVAL if result value is invalid or vp isn't a trigger vnode
 */
extern int vnode_trigger_update(vnode_t vp, resolver_result_t result);

struct vnode_trigger_info {
    trigger_vnode_resolve_callback_t    vti_resolve_func;
    trigger_vnode_unresolve_callback_t    vti_unresolve_func;
    trigger_vnode_rearm_callback_t        vti_rearm_func;
    trigger_vnode_reclaim_callback_t    vti_reclaim_func;
    void *                    vti_data;   /* auxiliary data (optional) */
    uint32_t                vti_flags;  /* optional flags (see below) */
};

/*
 * SPI for creating a trigger vnode
 *
 * Uses the VNCREATE_TRIGGER flavor with existing vnode_create() KPI
 *
 * Only one resolver per vnode.
 *
 * ERRORS (in addition to vnode_create errors):
 *    EINVAL (invalid resolver info, like invalid flags)
 *    ENOTDIR (only directories can have a resolver)
 *    EPERM (vnode cannot be a trigger - eg root dir of a file system)
 *    ENOMEM
 */
struct vnode_trigger_param {
    struct vnode_fsparam            vnt_params; /* same as for VNCREATE_FLAVOR */
    trigger_vnode_resolve_callback_t    vnt_resolve_func;
    trigger_vnode_unresolve_callback_t    vnt_unresolve_func;
    trigger_vnode_rearm_callback_t        vnt_rearm_func;
    trigger_vnode_reclaim_callback_t    vnt_reclaim_func;
    void *                    vnt_data;   /* auxiliary data (optional) */
    uint32_t                vnt_flags;  /* optional flags (see below) */
};

#define VNCREATE_TRIGGER    (('T' << 8) + ('V'))
#define VNCREATE_TRIGGER_SIZE    sizeof(struct vnode_trigger_param)

/*
 * vnode trigger flags (vnt_flags)
 *
 * VNT_AUTO_REARM:
 * On unmounts of a trigger mount, automatically re-arm the trigger.
 *
 * VNT_NO_DIRECT_MOUNT:
 * A trigger vnode instance that doesn't directly trigger a mount,
 * instead it triggers the mounting of sub-trigger nodes.
 */
#define VNT_AUTO_REARM        (1 << 0)
#define VNT_NO_DIRECT_MOUNT    (1 << 1)
#define VNT_VALID_MASK        (VNT_AUTO_REARM | VNT_NO_DIRECT_MOUNT)


/*
 * Information kept for each trigger vnode.
 */
typedef struct trigger_info trigger_info_t;

extern trigger_info_t *trigger_new(struct vnode_trigger_param *vnt,
                                   void *(*get_mount_args)(vnode_t, vfs_context_t, int *),
                                   void (*rel_mount_args)(void *));
extern void trigger_free(trigger_info_t *ti);

/*
 * The arguments passed to the thread routine that calls up to automountd
 * to resolve the trigger are assumed to be in a structure the first
 * element of which is one of these structures.
 */
struct trigger_callargs {
    vnode_t        tc_vp;        /* trigger vnode */
    fsid_t        tc_this_fsid;    /* fsid of file system with trigger vnode */
    trigger_info_t    *tc_ti;        /* trigger information */
    thread_t    tc_origin;    /* thread that fired up this thread */
    /* used for debugging purposes */
    uid_t        tc_uid;         /* Calling threads uid */
    au_asid_t    tc_asid;    /* and audit session id. */
    /* The two above fields are so that we can set a worker */
    /* processes to have the same uid and audit session */
    /* and thus be in the right security session for authentication */
    fsid_t        tc_mounted_fsid;/* fsid of newly-mounted file system */
    uint32_t    tc_retflags;    /* assorted MOUNT_RETF_ flags */
};

/*
 * Arguments passed to thread routine to call up to automountd to mount
 * an arbitrary URL on an arbitrary path.
 */
struct mount_url_callargs {
    struct trigger_callargs muc_t;    /* common args */
    char        *muc_url;    /* URL to mount */
    char        *muc_mountpoint; /* where to mount it */
    char        *muc_opts;    /* mount options to use; null string if none */
};

#define muc_this_fsid        muc_t.tc_this_fsid
#define muc_origin        muc_t.tc_origin
#define muc_uid            muc_t.tc_uid
#define muc_asid        muc_t.tc_asid
#define muc_mounted_fsid    muc_t.tc_mounted_fsid
#define muc_retflags        muc_t.tc_retflags

/*
 * Make an upcall to automountd to call SMBRemountServer() from the
 * SMB client framework.
 */
extern int SMBRemountServer(const void *ptr, size_t len, au_asid_t asid);


